% This function calculates the catchment-average time serieses (in Eq.7),
% the storm-average maps (in Eq.8) and the catchment- and storm-average
% terms (in Eq.9).

% TS (catchment-average time serieses in Eq.7):
% 1)[P]a, 2)[W1]a, 3)[W2]a, 4)[W]a, 5)[P]a[W1]a, 6)[P]a[W2]a, 7)[P]a[W]a,
% 8){P,W1}a, 9){P,W2}a, 10){P,W}a, 11)[R1]a, 12)[R2]a, 13)[R]a.

% AM (storm-average maps in Eq.8):
% 1)[P]t, 2)[W1]t, 3)[W2]t, 4)[W]t, 5)[P]t[W1]t, 6)[P]t[W2]t, 7)[P]t[W]t,
% 8){P,W1}t, 9){P,W2}t, 10){P,W}t, 11)[R1]t, 12)[R2]t,  13)[R]t.

% Sc (catchment- and storm-average terms in Eq.9):
% 1)[P]at; 2)[W]at; 3)R1; 4)R2; 5)R3; 6)R4; 7)MV; 8)[R]at.

% mask: mask of the basin;
% P: precipitation field for a rainfall event over the catchment area;
% WO/WI: surface/subsurface runoff coefficient field for a rainfall event
%        over the catchment area.

function [TS,AM,Sc]=AFV(mask,P,WO,WI)

Area=find(mask==1);

% Equation 7
Pa=nan(size(P,3),1);
WIa=nan(size(P,3),1);
WOa=nan(size(P,3),1);
ca_P_WI=nan(size(P,3),1);
ca_P_WO=nan(size(P,3),1);

for h=1:size(P,3)
  pm=P(:,:,h);
  Pa(h)=sum(pm(Area))/length(Area); % [P]a

  wim=WI(:,:,h);
  wom=WO(:,:,h);
  WIa(h)=sum(wim(Area))/length(Area); % [W1]a
  WOa(h)=sum(wom(Area))/length(Area); % [W2]a

  cc=cov(pm(Area),wim(Area),1);
  ca_P_WI(h)=cc(2,1); % {P,W1}a
  cc=cov(pm(Area),wom(Area),1);
  ca_P_WO(h)=cc(2,1); % {P,W2}a
end

Wa=[WIa WOa WIa+WOa]; % [W]a
PaWa=[Pa.*WIa Pa.*WOa Pa.*WIa+Pa.*WOa]; % [P]a*[W]a
ca_P_W=[ca_P_WI ca_P_WO ca_P_WI+ca_P_WO]; % {P,W}a
Ra=PaWa+ca_P_W; % [R]a

% Equation 8
Pt=nan(size(mask,1),size(mask,2));
WIt=nan(size(mask,1),size(mask,2));
WOt=nan(size(mask,1),size(mask,2));

ct_P_WI=nan(size(mask,1),size(mask,2)); % {P,W1}t
ct_P_WO=nan(size(mask,1),size(mask,2)); % {P,W2}t

for a=1:length(Area)
  y=ceil(Area(a)/size(mask,1));
  x=Area(a)-(y-1)*size(mask,1);

  pt=reshape(P(x,y,:),size(P,3),1);
  wit=reshape(WI(x,y,:),size(P,3),1);
  wot=reshape(WO(x,y,:),size(P,3),1);

  Pt(x,y)=sum(pt)/length(pt);
  WIt(x,y)=sum(wit)/length(wit);
  WOt(x,y)=sum(wot)/length(wot);

  cc=cov(pt,wit,1);
  ct_P_WI(x,y)=cc(2,1); % {P,W1}t
  cc=cov(pt,wot,1);
  ct_P_WO(x,y)=cc(2,1); % {P,W2}t
end

Wt=cat(3,WIt,WOt,WIt+WOt); % [W]t
PtWt=cat(3,Pt.*WIt,Pt.*WOt,Pt.*WIt+Pt.*WOt); % [P]t*[W]t
ct_P_W=cat(3,ct_P_WI,ct_P_WO,ct_P_WI+ct_P_WO); % {P,W}t
Rt=PtWt+ct_P_W;

% Equation 9
Pat=nanmean(Pa); % [P]at
Wat=[mean(WIa) mean(WOa) mean(WIa)+mean(WOa)]; % [W]at

cc=cov(Pa,WIa,1);
ct_Pa_WIa=cc(2,1); % {[P]a,[W1]a}t
cc=cov(Pa,WOa,1);
ct_Pa_WOa=cc(2,1); % {[P]a,[W2]a}t
ct_Pa_Wa=[ct_Pa_WIa ct_Pa_WOa ct_Pa_WIa+ct_Pa_WOa]; % {[P]a,[W]a}t

cc=cov(Pt(Area),WIt(Area),1);
ca_Pt_WIt=cc(2,1); % {[P]t,[W1]t}a
cc=cov(Pt(Area),WOt(Area),1);
ca_Pt_WOt=cc(2,1); % {[P]t,[W2]t}a
ca_Pt_Wt=[ca_Pt_WIt ca_Pt_WOt ca_Pt_WIt+ca_Pt_WOt]; % {[P]t,[W]t}a

ca_PmPt_WImWIt=nan(size(P,3),1);
ca_PmPt_WOmWOt=nan(size(P,3),1);
for h=1:size(P,3)
  pmpt=P(:,:,h)-Pt;
  wimwit=WI(:,:,h)-WIt;
  cc=cov(pmpt(Area),wimwit(Area),1);
  ca_PmPt_WImWIt(h,1)=cc(2,1);
  womwot=WO(:,:,h)-WOt;
  cc=cov(pmpt(Area),womwot(Area),1);
  ca_PmPt_WOmWOt(h,1)=cc(2,1);
end
et_ca_PmPt_WmWt=[mean(ca_PmPt_WImWIt) mean(ca_PmPt_WOmWOt)...
    mean(ca_PmPt_WImWIt)+mean(ca_PmPt_WOmWOt)]; % [{P-[P]t,W-[W]t}a]t

% Movement
R1=repmat(Pat,1,3).*Wat;
R2=ct_Pa_Wa;
R3=ca_Pt_Wt;
R4=et_ca_PmPt_WmWt;

Rat=R1+R2+R3+R4; % [R]at
MV=R4-R2.*R3./R1; % Movement

% Output
TS=[Pa Wa PaWa ca_P_W Ra]; % T by 13
AM=cat(3,Pt,Wt,PtWt,ct_P_W,Rt); % A by 13
Sc=[repmat(Pat,1,3);Wat;R1;R2;R3;R4;MV;Rat]; % 8 by 3
end
